<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\TherapistCommunityCategoriesExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\TherapistCommunityCategories\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\TherapistCommunityCategoriesImport;
use App\Models\TherapistCommunityCategory;
use App\Models\TherapistCommunityMainCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;

class TherapistCommunityCategoriesController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist_community_category.index');
        $this->middleware('permission:therapist_community_category.add', ['only' => ['store']]);
        $this->middleware('permission:therapist_community_category.edit', ['only' => ['update']]);
        $this->middleware('permission:therapist_community_category.delete', ['only' => ['destroy']]);
        $this->middleware('permission:therapist_community_category.export', ['only' => ['export']]);
        $this->middleware('permission:therapist_community_category.import', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null)
    {
        if ($req != null) {
            $therapist_community_categories =  TherapistCommunityCategory::withAll();
            if ($req->trash && $req->trash == 'with') {
                $therapist_community_categories =  $therapist_community_categories->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_community_categories =  $therapist_community_categories->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_community_categories = $therapist_community_categories->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_community_categories = $therapist_community_categories->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_community_categories = $therapist_community_categories->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_community_categories = $therapist_community_categories->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_community_categories = $therapist_community_categories->get();
                return $therapist_community_categories;
            }
            $therapist_community_categories = $therapist_community_categories->get();
            return $therapist_community_categories;
        }
        $therapist_community_categories = TherapistCommunityCategory::withAll()->orderBy('id', 'desc')->get();
        return $therapist_community_categories;
    }


    /*********View All TherapistCommunityCategories  ***********/
    public function index(Request $request)
    {
        $therapist_community_categories = $this->getter($request);
        return view('super_admins.therapist_community_categories.index')->with('therapist_community_categories', $therapist_community_categories);
    }

    /*********View Create Form of TherapistCommunityCategory  ***********/
    public function create()
    {
        $therapist_community_main_categories = TherapistCommunityMainCategory::active()->get();
        return view('super_admins.therapist_community_categories.create', compact('therapist_community_main_categories'));
    }

    /*********Store TherapistCommunityCategory  ***********/
    public function store(CreateRequest $request)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $data['image'] = uploadCroppedFile($request, 'image', 'therapist_community_categories');

            $therapist_community_category = TherapistCommunityCategory::create($data);
            $therapist_community_category->slug = Str::slug($therapist_community_category->name . ' ' . $therapist_community_category->id, '-');
            $therapist_community_category->save();
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_community_categories.index')->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_community_categories.index')->with('message', 'TherapistCommunityCategory Created Successfully')->with('message_type', 'success');
    }

    /*********View TherapistCommunityCategory  ***********/
    public function show(TherapistCommunityCategory $therapist_community_category)
    {
        return view('super_admins.therapist_community_categories.show', compact('therapist_community_category'));
    }

    /*********View Edit Form of TherapistCommunityCategory  ***********/
    public function edit(TherapistCommunityCategory $therapist_community_category)
    {
        $therapist_community_main_categories = TherapistCommunityMainCategory::active()->get();
        return view('super_admins.therapist_community_categories.edit', compact('therapist_community_category', 'therapist_community_main_categories'));
    }

    /*********Update TherapistCommunityCategory  ***********/
    public function update(CreateRequest $request, TherapistCommunityCategory $therapist_community_category)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if ($request->image) {
                $data['image'] = uploadCroppedFile($request, 'image', 'therapist_community_categories', $therapist_community_category->image);
            } else {
                $data['image'] = $therapist_community_category->image;
            }
            $therapist_community_category->update($data);
            $therapist_community_category = TherapistCommunityCategory::find($therapist_community_category->id);
            $slug = Str::slug($therapist_community_category->name . ' ' . $therapist_community_category->id, '-');
            $therapist_community_category->update([
                'slug' => $slug
            ]);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_community_categories.index')->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_community_categories.index')->with('message', 'TherapistCommunityCategory Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_community_categories = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_community_categories." . $extension;
        return Excel::download(new TherapistCommunityCategoriesExport($therapist_community_categories), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new TherapistCommunityCategoriesImport, $file);
        return redirect()->back()->with('message', 'Blog Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE TherapistCommunityCategory ***********/
    public function destroy(TherapistCommunityCategory $therapist_community_category)
    {
        $therapist_community_category->delete();
        return redirect()->back()->with('message', 'TherapistCommunityCategory Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE TherapistCommunityCategory ***********/
    public function destroyPermanently(Request $request, $therapist_community_category)
    {
        $therapist_community_category = TherapistCommunityCategory::withTrashed()->find($therapist_community_category);
        if ($therapist_community_category) {
            if ($therapist_community_category->trashed()) {
                if ($therapist_community_category->image && file_exists(public_path($therapist_community_category->image))) {
                    unlink(public_path($therapist_community_category->image));
                }
                $therapist_community_category->forceDelete();
                return redirect()->back()->with('message', 'Blog Category Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Blog Category is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Blog Category Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore TherapistCommunityCategory***********/
    public function restore(Request $request, $therapist_community_category)
    {
        $therapist_community_category = TherapistCommunityCategory::withTrashed()->find($therapist_community_category);
        if ($therapist_community_category->trashed()) {
            $therapist_community_category->restore();
            return redirect()->back()->with('message', 'Blog Category Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Blog Category Not Found')->with('message_type', 'error');
        }
    }
}
